// Copyright (C) DEWETRON GmbH 2020

#include "dt_stopwatch.h"

#ifdef WIN32
#include <Windows.h>

#define US_PER_SECOND 1000000

typedef struct 
{
    LARGE_INTEGER m_time_start;
    LARGE_INTEGER m_time_freq;
    uint64_t m_last_time;
} DT_StopWatchHandleImp;

void DT_StopWatch_Create(DT_StopWatchHandle* sw)
{
    DT_StopWatchHandleImp* handle_imp = new DT_StopWatchHandleImp;
    handle_imp->m_time_start.QuadPart = 0;
    handle_imp->m_last_time = 0;
    QueryPerformanceFrequency(&handle_imp->m_time_freq);
    *sw = handle_imp;
}

void DT_StopWatch_Destroy(DT_StopWatchHandle* sw)
{
    DT_StopWatchHandleImp* handle_imp = (DT_StopWatchHandleImp*)(*sw);
    delete handle_imp;
}

void DT_StopWatch_Start(DT_StopWatchHandle* sw)
{
    DT_StopWatchHandleImp* handle_imp = (DT_StopWatchHandleImp*)(*sw);
    QueryPerformanceCounter(&handle_imp->m_time_start);
}

void DT_StopWatch_Stop(DT_StopWatchHandle* sw)
{
    DT_StopWatchHandleImp* handle_imp = (DT_StopWatchHandleImp*)(*sw);
    LARGE_INTEGER time_end;
    QueryPerformanceCounter(&time_end);
    if (handle_imp->m_time_freq.QuadPart != 0)
    {
        uint64_t time_diff = time_end.QuadPart - handle_imp->m_time_start.QuadPart;
        if (handle_imp->m_time_freq.QuadPart > US_PER_SECOND)
        {
            time_diff = (time_diff * US_PER_SECOND) / handle_imp->m_time_freq.QuadPart;
        }
        handle_imp->m_last_time = time_diff;
    }
}

uint64_t DT_StopWatch_GetMeantimeUS(DT_StopWatchHandle* sw)
{
    DT_StopWatchHandleImp* handle_imp = (DT_StopWatchHandleImp*)(*sw);
    LARGE_INTEGER time_end;
    QueryPerformanceCounter(&time_end);
    if (handle_imp->m_time_freq.QuadPart != 0)
    {
        uint64_t time_diff = time_end.QuadPart - handle_imp->m_time_start.QuadPart;
        if (handle_imp->m_time_freq.QuadPart > US_PER_SECOND)
        {
            time_diff = (time_diff * US_PER_SECOND) / handle_imp->m_time_freq.QuadPart;
        }
        return time_diff;
    }
    return 0;
}

uint64_t DT_StopWatch_GetMeantimeMS(DT_StopWatchHandle* sw)
{
    return DT_StopWatch_GetMeantimeUS(sw) / 1000;
}

uint64_t DT_StopWatch_GetUS(DT_StopWatchHandle* sw)
{
    DT_StopWatchHandleImp* handle_imp = (DT_StopWatchHandleImp*)(*sw);
    return handle_imp->m_last_time;
}

uint64_t DT_StopWatch_GetMS(DT_StopWatchHandle* sw)
{
    DT_StopWatchHandleImp* handle_imp = (DT_StopWatchHandleImp*)(*sw);
    return handle_imp->m_last_time / 1000;
}

#else

#include <time.h>
#include <string.h>

typedef struct 
{
    timespec m_time_start;
    timespec m_time_freq;
    uint64_t m_last_time;
} DT_StopWatchHandleImp;

void DT_StopWatch_Create(DT_StopWatchHandle* sw)
{
    DT_StopWatchHandleImp* handle_imp = new DT_StopWatchHandleImp;
    memset(&handle_imp->m_time_start, 0, sizeof(timespec));
    handle_imp->m_last_time = 0;
    clock_getres(CLOCK_MONOTONIC, &handle_imp->m_time_freq);
    *sw = handle_imp;
}

void DT_StopWatch_Destroy(DT_StopWatchHandle* sw)
{
    DT_StopWatchHandleImp* handle_imp = (DT_StopWatchHandleImp*)(*sw);
    delete handle_imp;
}

void DT_StopWatch_Start(DT_StopWatchHandle* sw)
{
    DT_StopWatchHandleImp* handle_imp = (DT_StopWatchHandleImp*)(*sw);
    clock_gettime(CLOCK_MONOTONIC, &handle_imp->m_time_start);
}

void DT_StopWatch_Stop(DT_StopWatchHandle* sw)
{
    DT_StopWatchHandleImp* handle_imp = (DT_StopWatchHandleImp*)(*sw);
    timespec time_end;
    clock_gettime(CLOCK_MONOTONIC, &time_end);
    handle_imp->m_last_time = (time_end.tv_sec - handle_imp->m_time_start.tv_sec) * 1000000 + (time_end.tv_nsec - handle_imp->m_time_start.tv_nsec) / 1000;
}

uint64_t DT_StopWatch_GetMeantimeUS(DT_StopWatchHandle* sw)
{
    DT_StopWatchHandleImp* handle_imp = (DT_StopWatchHandleImp*)(*sw);
    timespec time_end;
    clock_gettime(CLOCK_MONOTONIC, &time_end);
    return (time_end.tv_sec - handle_imp->m_time_start.tv_sec) * 1000000 + (time_end.tv_nsec - handle_imp->m_time_start.tv_nsec) / 1000;
}

uint64_t DT_StopWatch_GetMeantimeMS(DT_StopWatchHandle* sw)
{
    return DT_StopWatch_GetMeantimeUS(sw) / 1000;
}

uint64_t DT_StopWatch_GetUS(DT_StopWatchHandle* sw)
{
    DT_StopWatchHandleImp* handle_imp = (DT_StopWatchHandleImp*)(*sw);
    return handle_imp->m_last_time;
}

uint64_t DT_StopWatch_GetMS(DT_StopWatchHandle* sw)
{
    return DT_StopWatch_GetUS(sw) / 1000;
}

#endif