// Copyright DEWETRON GmbH 2018

#include "dt_cmdline.h"
#include "dt_log.h"
#include <algorithm>
#include <cctype>
#include <iterator>
#include <sstream>
#include <vector>

namespace
{
    enum OptionType
    {
        SWITCH,
        VALUE
    };

    struct Option
    {
        std::string long_option;
        std::string short_option;
        OptionType  option_type;
        std::string value;
        std::string description;
    };

    // trim from start (in place)
    static inline void ltrim(std::string &s) {
        s.erase(s.begin(), std::find_if(s.begin(), s.end(), [](int ch) {
            return !std::isspace(ch);
    }));
    }

    // trim from end (in place)
    static inline void rtrim(std::string &s) {
        s.erase(std::find_if(s.rbegin(), s.rend(), [](int ch) {
            return !std::isspace(ch);
    }).base(), s.end());
    }

    // trim from both ends (in place)
    static inline void trim(std::string &s) {
        ltrim(s);
        rtrim(s);
    }
}

void printUsage(const std::vector<Option> options)
{
    std::cout << "dt_client [OPTION]\n";
    std::cout << "Example OXYGEN data transfer client to connect to OXYGEN data transfer plugin\n";

    std::cout << "Arguments:" << std::endl;

    int max_tabcount = 0;

    // prerender
    for (const auto& opt : options)
    {
        std::stringstream ss;

        if (!opt.short_option.empty() && !opt.long_option.empty())
        {
            ss << "  -" << opt.short_option << ", --" << opt.long_option;
        }
        else if (!opt.short_option.empty())
        {
            ss << "  -" << opt.short_option;
        }
        else if (!opt.long_option.empty())
        {
            ss << "      --" << opt.long_option;
        }

        if (opt.option_type == VALUE)
        {
            ss << " VALUE";
        }

        int tabcount = static_cast<int>(ss.str().size() / 8);
        max_tabcount = std::max(max_tabcount, tabcount);
    }

    // render
    for (const auto& opt : options)
    {
        std::stringstream ss;

        if (!opt.short_option.empty() && !opt.long_option.empty())
        {
            ss << "  -" << opt.short_option << ", --" << opt.long_option;
        }
        else if (!opt.short_option.empty())
        {
            ss << "  -" << opt.short_option;
        }
        else if (!opt.long_option.empty())
        {
            ss << "      --" << opt.long_option;
        }

        if (opt.option_type == VALUE)
        {
            ss << " VALUE";
        }

        int tabcount = static_cast<int>(ss.str().size() / 8);

        int tabs_to_fill = max_tabcount - tabcount + 1;

        ss << std::string(tabs_to_fill, '\t');
        ss << opt.description << " " << "\n";
        std::cout << ss.str();
    }
}


void parseOptionValue(Option& opt, std::vector<std::string>::iterator& arg, std::vector<std::string>& arguments)
{
    if (opt.long_option == "help")
    {
        throw std::exception();
    }

    switch (opt.option_type)
    {
    case SWITCH:
        {
            opt.value = std::to_string(std::stoi(opt.value) + 1);
            ++arg;
        }
        return;
    case VALUE: break;
    default: return;
    }

    // possible
    // (1) opt=1
    // (2) opt= 1

    // (3) opt 1
    // (4) opt =1
    // (5) opt = 1

    auto token = *arg;
    trim(token);

    auto pos = token.find_first_of("=");
    if (pos != std::string::npos)
    {
        if (pos + 1 < token.size())
        {
            // (1)
            opt.value = token.substr(pos + 1, token.size() - pos - 1);
            ++arg;
        }
        else
        {
            // (2)
            ++arg;
            if (arg == arguments.end()) throw std::runtime_error("No value for option: " + opt.long_option);

            if (arg->at(0) != '-')
            {
                opt.value = *arg;
                ++arg;
            }
            else
            {
                throw std::runtime_error("Invalid value for option: " + opt.long_option);
            }
        }
        return;
    }
    else
    {
        // remove leading "-"
        auto pos = token.find_first_not_of('-');
        token = token.substr(pos);

        if ((opt.long_option == token) || (opt.short_option == token))
        {
            // (3)(4)(5)
            ++arg;
            if (arg == arguments.end()) throw std::runtime_error("No value for option: " + opt.long_option);

            switch(arg->at(0))
            {
            case '-':
                throw std::runtime_error("No value given for option: " + opt.long_option);
                break;

            case '=':
                if (arg->size() == 1)
                {
                    // (5)
                    ++arg;
                    if (arg == arguments.end()) throw std::runtime_error("No value for option: " + opt.long_option);

                    if (arg->at(0) != '-')
                    {
                        opt.value = *arg;
                        ++arg;
                    }
                    else
                    {
                        throw std::runtime_error("Invalid value for option: " + opt.long_option);
                    }
                }
                else
                {
                    // (4)
                    opt.value = arg->substr(1);
                    ++arg;
                }
                break;
            default:
                // (3)
                opt.value = *arg;
                ++arg;
                break;
            }
        }
    }
}

bool parseAruments(const int argc, char* argv[], DtClientConfig& config)
{
    std::vector<Option> options {
            {"help",        "h", SWITCH,    "0",          "Show this message"}
            , {"address",   "a", VALUE,     "127.0.0.1",  "IP of data stream server"}
            , {"port",      "p", VALUE,     "5555",       "Port of data stream server"}
            , {"print_samples", "", SWITCH, "0",          "Print received data samples"}
            , {"perf",      "",  SWITCH,    "0",          "Print data transfer performance"}
            , {"verbose",   "v", SWITCH,    "0",          "Be more verbose"}
        };

    try
    {
        // parse
        std::vector<std::string> arguments;
        for (int i = 1; i < argc; ++i)
        {
            arguments.push_back(argv[i]);
        }

        auto arg = arguments.begin();
        while (arg != arguments.end())
        {
            std::string token = *arg;
            if (token.substr(0, 2) == "--")         // long option
            {
                auto rt = token.substr(2);
                bool handled = false;
                for (auto& opt : options)
                {
                    if (opt.long_option.size() >= rt.size())
                    {
                        if (opt.long_option.find(rt) != std::string::npos)
                        {
                            parseOptionValue(opt, arg, arguments);
                            handled = true;
                        }
                    }
                    else
                    {
                        if (rt.find(opt.long_option) != std::string::npos)
                        {
                            parseOptionValue(opt, arg, arguments);
                            handled = true;
                        }
                    }
                }
                if (!handled)
                {
                    std::string err = std::string("Unknown long option: " + token);
                    throw std::runtime_error(err);
                }
            }
            else if (token.substr(0, 1) == "-")     // short option
            {
                if (token.size() != 2)
                {
                    std::string err = std::string("Invalid short option: " + token);
                    throw std::runtime_error(err);
                }

                auto rt = token.substr(1);  // strip leading '-'
                bool handled = false;
                for (auto& opt : options)
                {
                    if (opt.short_option == rt)
                    {
                        parseOptionValue(opt, arg, arguments);
                        handled = true;
                    }
                }
                if (!handled)
                {
                    std::string err = std::string("Unknown short option: " + token);
                    throw std::runtime_error(err);
                }
            }
            else
            {
                // value or positional option
                std::string err = std::string("Unknown argument: " + token);
                throw std::runtime_error(err);
            }
        }


        for (const auto& opt : options)
        {
            if (opt.long_option == "address")
            {
                config.m_address = opt.value;
            }
            else if (opt.long_option == "port")
            {
                config.m_port = std::stoul(opt.value);
            }
            else if (opt.long_option == "print_samples")
            {
                config.m_print_samples = std::stoi(opt.value);
            }
            else if (opt.long_option == "perf")
            {
                config.m_performance = std::stoi(opt.value);
            }
            else if (opt.long_option == "verbose")
            {
                config.m_verbose = std::stoi(opt.value);
            }
        }

        DTLOG_VERBOSITY(config.m_verbose);

#ifndef NDEBUG
        for (const auto opt : options)
        {
            DTLOG_INFO(3, "  -" << opt.short_option << ", --" << opt.long_option << " " << opt.value);
        }
#endif


    }
    catch (const std::runtime_error& e)
    {
        std::cerr << "Error: " << e.what() << "\n" << std::endl;
        printUsage(options);
        return false;
    }
    catch (const std::exception&)
    {
        printUsage(options);
        return false;
    }

    return true;
}
