// Copyright DEWETRON GmbH 2018
#pragma once

#include <inttypes.h>

/**
 * Protocol 1.5 defines
 */

#define DT_PROTOCOL_VERSION         0x01050000
#define DT_WELCOME_MSG_SIZE         64

#define DT_START_TOKEN              "OXYGEN<<"
#define DT_END_TOKEN                ">>OXYGEN"
#define DT_START_TOKEN_SIZE         8
#define DT_END_TOKEN_SIZE           8

// Fix size packets:
#define DT_PACKET_BASE_SIZE         8
#define DT_PACKET_HEADER_SIZE       12
#define DT_PACKET_INFO_SIZE         32
#define DT_PACKET_ACQ_INFO_SIZE     16
#define DT_PACKET_FOOTER_SIZE       20

#define DT_PACKET_SYNC_HEADER_SIZE    36
#define DT_PACKET_ASYNC_HEADER_SIZE   28


/**
 * Sub Packet types valid for version 1.5
 */
enum SubPacketType
{
    SBT_PACKET_INFO         = 0x00000001,
    SBT_XML_CONFIG          = 0x00000002,
    SBT_SYNC_FIXED          = 0x00000003,
    SBT_SYNC_VARIABLE       = 0x00000004,
    SBT_ASYNC_FIXED         = 0x00000005,
    SBT_ASYNC_VARIABLE      = 0x00000006,
    SBT_PACKET_FOOTER       = 0x00000007,
};

/**
 * Mark the dedication of the stream
 */
enum StreamStatus
{
    ST_FIRST_PACKET         = 0x00000001,
    ST_LAST_PACKET          = 0x00000002,
    ST_NORMAL_PACKET        = 0x00000000,
    ST_ERROR_PACKET         = 0x10000000,
};

/**
 * Sample data types
 */
enum DtDataType
{
    dt_sint8 = 0,          // 0
    dt_uint8,              // 1
    dt_sint16,             // 2
    dt_uint16,             // 3
    dt_sint24,             // 4
    dt_uint24,             // 5
    dt_sint32,             // 6
    dt_uint32,             // 7
    dt_sint64,             // 8
    dt_uint64,             // 9
    dt_float,              // 10
    dt_double,             // 11
    dt_complex_float,      // 12
    dt_complex_double,     // 13
    dt_string,             // 14
    dt_binary,             // 15
    dt_CAN,                // 16
};


/**
 * Packet info subpacket V1.5
 * "SBT_PACKET_INFO"
 */
typedef struct _DtPacketInfo
{
    uint32_t protocol_version;
    uint32_t stream_id;
    uint32_t sequence_number;
    uint32_t stream_status;
    uint32_t seed;
    uint32_t number_of_subpackets;
} DtPacketInfo;

/**
 * Generic XML subpacket V1.5
 * "SBT_XML_CONFIG"
 */
typedef struct _DtXmlSubPacket
{
    char* xml_content;
    uint32_t xml_content_size;
} DtXmlSubPacket;


/**
 * "SBT_PACKET_FOOTER"
 */
typedef struct _DtPacketFooter
{
    uint32_t checksum;
} DtPacketFooter;


/**
 * Sync channel fixed sample size V1.5
 * "SBT_SYNC_FIXED"
 */
typedef struct _DtChannelSyncFixed
{
    uint32_t channel_data_type;     // fixed size samples (not dt_binary, dt_string)
    uint32_t channel_dimension;
    uint32_t number_samples;
    uint64_t timestamp;
    uint64_t timebase_frequency;
    uint8_t* sample_data;
    uint32_t sample_data_size;
} DtChannelSyncFixed;


/**
 * Sync channel variable sample size V1.5
 * "SBT_SYNC_VARIABLE"
 */
typedef struct _DtChannelSyncVariable
{
    uint32_t channel_data_type;     // fixed size samples (not dt_binary, dt_string)
    uint32_t channel_dimension;
    uint32_t number_samples;
    uint64_t timestamp;
    uint64_t timebase_frequency;
    uint8_t* sample_data;
    uint32_t sample_data_size;
} DtChannelSyncVariable;


/**
 * Async channel fixed sample size V1.5
 * "SBT_ASYNC_FIXED"
 */
typedef struct _DtChannelAsyncFixed
{
    uint32_t channel_data_type;     // fixed size samples (not dt_binary, dt_string)
    uint32_t channel_dimension;
    uint32_t number_samples;
    uint64_t timebase_frequency;
    uint8_t* sample_data;
    uint32_t sample_data_size;
} DtChannelAsyncFixed;


/**
 * Async channel variable sample size V1.5
 * "SBT_ASYNC_VARIABLE"
 */
typedef struct _DtChannelAsyncVariable
{
    uint32_t channel_data_type;     // fixed size samples (not dt_binary, dt_string)
    uint32_t channel_dimension;
    uint32_t number_samples;
    uint64_t timebase_frequency;
    uint8_t* sample_data;
    uint32_t sample_data_size;
} DtChannelAsyncVariable;


int32_t getDtDataTypeSize(uint32_t dt);

const char* getDtDataTypeName(uint32_t dt);
